import 'package:flutter/material.dart';
import 'package:course_app/utils/battle.dart';
import 'package:course_app/utils/database_helper.dart';

class BattleView extends StatelessWidget {
  BattleView();

  // Function to load battles from the database
  Future<List<Battle>> fetchBattles() async {
    final dbHelper = DatabaseHelper.instance;
    final battles = await dbHelper.getAllBattles();

    //print('Fetched ${battles.length} battles from the database'); // Add this line to check the number of battles
    return battles;
  }

  @override
  Widget build(BuildContext context) {
    return Expanded(
      child: Padding(
        padding: const EdgeInsets.symmetric(vertical: 8.0),
        child: FutureBuilder<List<Battle>>(
          future: fetchBattles(), // Load data using FutureBuilder
          builder: (context, snapshot) {
            if (snapshot.connectionState == ConnectionState.waiting) {
              // Show loading indicator while data is being fetched
              return Center(child: CircularProgressIndicator());
            } else if (snapshot.hasError) {
              // Show an error message if fetching data failed
              return Center(child: Text('Error: ${snapshot.error}'));
            } else if (snapshot.hasData) {
              final battles = snapshot.data ?? []; // Retrieve the loaded data

              return ListView.builder(
                itemCount: battles.length,
                itemBuilder: (context, index) {
                  final battle = battles[index];
                  return InkWell(
                    onTap: () {
                      Navigator.pushNamed(
                        context,
                        '/battle',
                        arguments: {'battle': battle},
                      );
                    },
                    child: Container(
                      margin: EdgeInsets.symmetric(vertical: 5, horizontal: 10),
                      padding: EdgeInsets.all(10),
                      decoration: BoxDecoration(
                        color: Colors.grey[200],
                        border: Border.all(color: Colors.grey),
                        borderRadius: BorderRadius.circular(8),
                      ),
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Text(
                            battle.name,
                            style: TextStyle(
                                fontWeight: FontWeight.bold, fontSize: 16),
                          ),
                          SizedBox(height: 5),
                          Text(
                            battle.date,
                            style: TextStyle(
                                fontSize: 14, fontStyle: FontStyle.italic),
                          ),
                        ],
                      ),
                    ),
                  );
                },
              );
            } else {
              // Show an empty state if there is no data available
              return Center(child: Text('No battles available.'));
            }
          },
        ),
      ),
    );
  }
}
