<?php
// Set the content type to JSON for all responses
header("Content-Type: application/json");

// Database connection details
$host = 'localhost';  // Hostname of the database server
$db_name = 'user_auth'; // Name of the database
$username = 'student';  // Database username
$password = '$tr0ngP4$$w0rd';  // Database password

// Create a new MySQLi connection
// The mysqli() function in PHP is part of the MySQLi (MySQL Improved) extension,
// which provides an interface for interacting with MySQL databases.
// Specifically, it is used to establish a connection to a MySQL database server.
// Syntax of mysqli()
// $conn = new mysqli($host, $username, $password, $dbname);
// Parameters
// - $host: The hostname or IP address of the MySQL server (e.g., 'localhost' or '127.0.0.1').
// - $username: The username used to connect to the MySQL database.
// - $password: The password for the MySQL user.
// - $dbname: The name of the database you want to connect to.
// Return Value
// - Success: Returns an instance of the mysqli class, which can be used to interact with the database (e.g., execute queries, fetch data).
// - Failure: The connect_error property of the returned object will contain an error message, and the connection will not be usable.
$conn = new mysqli($host, $username, $password, $db_name);

// Check if the database connection failed
if ($conn->connect_error) {
    http_response_code(500); // Set HTTP response code to 500 (Internal Server Error)
    echo json_encode(['success' => false, 'message' => 'Database connection failed']); // Send error message as JSON
    exit; // Exit the script
}

// Determine the HTTP method (GET, POST, PUT, DELETE, etc.) and the requested endpoint
// GET
// - Purpose: Retrieve data from a server.
// - Typical Use Case: Fetch a list of items or a specific resource.
// POST
// - Purpose: Submit data to a server to create a new resource.
// - Typical Use Case: Create a new resource (e.g., user registration, submitting a form).
// PUT
// - Purpose: Update or replace an existing resource.
// - Typical Use Case: Update the entire resource or replace it.
// DELETE
// - Purpose: Delete a resource on the server.
// - Typical Use Case: Remove a specific resource.
// If the user hits the register button in the flutter app, the request is going to be POST
// i.e. $method = 'POST'
$method = $_SERVER['REQUEST_METHOD']; // Get the HTTP method of the request

// Assuming the user sends a request '$serverIp/$authApi/register'
// e.g. http://192.168.1.10/auth_api.php/register
// $_SERVER['PATH_INFO'] = '/register'
$endpoint = explode('/', trim($_SERVER['PATH_INFO'], '/'))[0]; // Extract the first segment of the URL path

// Read the JSON body of the request and decode it into an associative array (map)
$data = json_decode(file_get_contents("php://input"), true);

// Route the request based on the endpoint
switch ($endpoint) {
    // In the case of registration, this is the part that gets through
    case 'register': // If the endpoint is 'register'
        if ($method === 'POST') { // Only allow POST method for registration
            registerUser($conn, $data); // Call the registerUser function
        } else {
            sendError(405, 'Method Not Allowed'); // Send 405 error for unsupported methods
        }
        break;
    // In the case of login, this is the part that gets through
    case 'login': // If the endpoint is 'login'
        if ($method === 'POST') { // Only allow POST method for login
            loginUser($conn, $data); // Call the loginUser function
        } else {
            sendError(405, 'Method Not Allowed'); // Send 405 error for unsupported methods
        }
        break;

    case 'profile': // If the endpoint is 'profile'
        if ($method === 'GET') { // Allow GET method to fetch profile
            fetchUserProfile($conn, $_GET['email']); // Fetch the user profile using the provided email
        } elseif ($method === 'PUT') { // Allow PUT method to update profile
            updateUserProfile($conn, $data); // Update the user profile
        } else {
            sendError(405, 'Method Not Allowed'); // Send 405 error for unsupported methods
        }
        break;

    default: // If the endpoint is not recognized
        sendError(404, 'Endpoint Not Found'); // Send 404 error
        break;
}

// Close the database connection
$conn->close();

// Function to handle user registration
function registerUser($conn, $data) {
    // Check if required fields are present in the request
    if (!isset($data['username'], $data['email'], $data['password'])) {
        sendError(400, 'Invalid Input'); // Send 400 error if inputs are missing
    }

    // Extract and process user input
    $username = $data['username'];
    $email = $data['email'];
    $password = password_hash($data['password'], PASSWORD_BCRYPT); // Hash the password securely

    // Prepare SQL statement to insert user data
    // This protects against injection attacks
    $stmt = $conn->prepare("INSERT INTO users (username, email, password) VALUES (?, ?, ?)");
    $stmt->bind_param("sss", $username, $email, $password); // Bind parameters to the statement

    // Execute the statement and send a response
    if ($stmt->execute()) {
        echo json_encode(['success' => true, 'message' => 'User registered successfully']);
    } else {
        sendError(500, 'Database error: ' . $conn->error); // Send 500 error if the query fails
    }

    $stmt->close(); // Close the statement
}

// Function to handle user login
function loginUser($conn, $data) {
    // Check if required fields are present in the request
    if (!isset($data['email'], $data['password'])) {
        sendError(400, 'Invalid Input'); // Send a 400 Bad Request error if email or password is missing
    }

    // Extract user input
    $email = $data['email']; // Get the email from the client-provided data
    $password = $data['password']; // Get the password from the client-provided data

    // Prepare SQL statement to fetch the stored hashed password
    $stmt = $conn->prepare("SELECT password FROM users WHERE email = ?");
    // Bind the email parameter to the prepared SQL statement
    $stmt->bind_param("s", $email); // "s" indicates the type (string) of the bound parameter
    $stmt->execute(); // Execute the SQL query
    $stmt->store_result(); // Store the result to allow row count and binding

    // Check if the user exists
    if ($stmt->num_rows > 0) {
        $stmt->bind_result($hash); // Bind the fetched hashed password to the variable $hash
        $stmt->fetch(); // Retrieve the result from the executed query
        if (password_verify($password, $hash)) {
            // Verify the plain text password provided by the user against the hashed password in the database
            echo json_encode(['success' => true, 'message' => 'Login successful']);
            // If the password matches, send a success response in JSON format
        } else {
            sendError(401, 'Invalid password');
            // Send a 401 Unauthorized error if the password does not match
        }
    } else {
        sendError(404, 'User not found');
        // Send a 404 Not Found error if no user exists with the provided email
    }

    $stmt->close(); // Close the prepared statement to free resources
}


// Function to fetch user profile
function fetchUserProfile($conn, $email) {
    // Check if email is provided
    if (!$email) {
        sendError(400, 'Email is required'); // Send 400 error if email is missing
    }

    // Prepare SQL statement to fetch user details
    $stmt = $conn->prepare("SELECT id, username, email, created_at FROM users WHERE email = ?");
    $stmt->bind_param("s", $email); // Bind the email parameter
    $stmt->execute();
    $result = $stmt->get_result();

    // Check if the user exists and send the response
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc(); // Fetch user details as an associative array
        echo json_encode(['success' => true, 'user' => $user]);
    } else {
        sendError(404, 'User not found'); // Send 404 error if user does not exist
    }

    $stmt->close(); // Close the statement
}

// Function to update user profile
function updateUserProfile($conn, $data) {
    // Check if required fields are present in the request
    if (!isset($data['email'], $data['username'])) {
        sendError(400, 'Invalid Input'); // Send 400 error if inputs are missing
    }

    // Extract user input
    $email = $data['email'];
    $username = $data['username'];

    // Prepare SQL statement to update user details
    $stmt = $conn->prepare("UPDATE users SET username = ? WHERE email = ?");
    $stmt->bind_param("ss", $username, $email); // Bind parameters

    // Execute the statement and send a response
    if ($stmt->execute()) {
        echo json_encode(['success' => true, 'message' => 'User profile updated']);
    } else {
        sendError(500, 'Database error: ' . $conn->error); // Send 500 error if the query fails
    }

    $stmt->close(); // Close the statement
}

// Helper function to send error responses
function sendError($code, $message) {
    http_response_code($code); // Set the HTTP response code
    echo json_encode(['success' => false, 'message' => $message]); // Send error message as JSON
    exit; // Exit the script
}
?>
