import 'dart:async';
import 'dart:io';
import 'package:path/path.dart';
import 'package:sqflite/sqflite.dart';
import 'package:course_app/utils/battle.dart';
import 'package:flutter/services.dart'; // Import services for rootBundle

class DatabaseHelper {
  final String myDatabaseName = "battles.db";
  final String myTableName = "battles";
  final String myDatabasePath = "lib/assets/data/battles.db";

  // Make this a singleton class
  DatabaseHelper._privateConstructor();
  static final DatabaseHelper instance = DatabaseHelper._privateConstructor();

  // Database reference
  static Database? _database;

  // Getter to initialize and retrieve the database
  Future<Database> get database async {
    if (_database != null) return _database!;
    _database = await _initDatabase();
    return _database!;
  }

  // Initialize the database
  Future<Database> _initDatabase() async {
    final dbPath = await getDatabasesPath();
    String path = join(dbPath, myDatabaseName);

    // Check if the database already exists
    if (!await File(path).exists()) {
      print('Database does not exist, copying from assets...');
      await _copyDatabaseFromAssets(path);
    } else {
      print('Database already exists at $path');
    }

    // Open the database
    return await openDatabase(
      path,
      version: 1,
      //onCreate: _onCreate,
    );
  }

/*
// Use this to overwrite the existing database
// Development mode only
Future<Database> _initDatabase() async {
  final dbPath = await getDatabasesPath();
  String path = join(dbPath, myDatabaseName);

  // Check if the database already exists
  if (await File(path).exists()) {
    print('Existing database found, deleting and copying a new version.');
    await File(path).delete(); // Delete the existing database file
  } else {
    print('No existing database found, copying from assets.');
  }

  // Copy the database from assets
  await _copyDatabaseFromAssets(path);

  // Open the database
  return await openDatabase(
    path,
    version: 1,
    //onCreate: _onCreate,
  );
}
*/

  Future<void> _copyDatabaseFromAssets(String path) async {
    try {
      print('Attempting to copy database from assets...');
      ByteData data = await rootBundle.load(myDatabasePath);
      List<int> bytes = data.buffer.asUint8List();

      File file = File(path);
      await file.writeAsBytes(bytes);
      print('Database copied successfully!');
    } catch (e) {
      print('Error copying database: $e');
    }
  }

  /*
  // Create the battles table
  Future<void> _onCreate(Database db, int version) async {
    await db.execute('''
      CREATE TABLE battles (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT,
        date TEXT,
        summary TEXT
      )
      ''');
  }
  */

  // Insert a new battle
  Future<int> insertBattle(Battle battle) async {
    final db = await database;
    return await db.insert(myTableName, battle.toMap());
  }

  // Fetch all battles
  Future<List<Battle>> getAllBattles() async {
    final db = await database;
    final maps = await db.query(myTableName);

    // Convert List<Map<String, dynamic>> to List<Battle>
    return List.generate(maps.length, (i) {
      return Battle.fromMap(maps[i]);
    });
  }

  Future<void> logBattleCount() async {
    final db = await database;
    final result = await db.rawQuery('SELECT COUNT(*) AS count FROM battles');
    print('Number of battles in database: ${result[0]['count']}');
  }

  Future<int> updateBattle(Battle battle) async {
    final db = await database;
    return await db.update(
      myTableName,
      battle.toMap(),
      where: 'id = ?',
      whereArgs: [battle.id], // Use the ID to find the specific battle
    );
  }
}
